unit UPieces;

{$mode objfpc}{$H+}

// This is needed because FPC shows wrong warnings for BlockRead :-/
{$WARN 5057 off : Local variable "$1" does not seem to be initialized}

interface

uses
  Dialogs,
  Classes, SysUtils, Nodes, Maths, Meshes;

const
  PieceMeshScale = 10;

type

  TPieceModel = class;

  { TPiece }

  TPiece = class(TNode)
  private
    FModel: TPieceModel;
    FTransform: TTransform;
    FMesh: TMesh;
    FWorldMatrix: TMatrix;
    procedure SetTransform(AValue: TTransform);
  protected
    procedure ParentChanged; override;
    procedure ChildAdded(AChild: TNode); override;
  public
    constructor Create; override;
    destructor Destroy; override;
    procedure LoadMesh(AFileName: string);
    function AddChildPiece(AFileName: string): TPiece;
    procedure UpdateWorldMatrix;
    property Model: TPieceModel read FModel;
    property WorldMatrix: TMatrix read FWorldMatrix;
    property Transform: TTransform read FTransform write SetTransform;
    property Mesh: TMesh read FMesh;
  end;

  { TKeyFrame }

  TKeyFrame = record
    Piece: TPiece;
    Rotation: TVector;
  end;

  { TFrame }

  TFrame = class
    IsKey: Boolean;
    IsEvent: Boolean;
    KeyFrames: array of TKeyFrame;
    RootPos: TVector;
    procedure CopyPoseFrom(Src: TFrame);
  end;

  { TAnimation }

  TAnimation = class
  private
    FName: string;
    FModel: TPieceModel;
    FFrames: array of TFrame;
    FLooping: Boolean;
    FSmooth: Boolean;
    FFPS: Integer;
    function GetFrame(AIndex: Integer): TFrame;
    function GetFrames: Integer;
    function AddNewFrame: TFrame;
    procedure SetFrames(AValue: Integer);
    procedure SetLooping(AValue: Boolean);
    procedure SetSmooth(AValue: Boolean);
  public
    constructor Create(AModel: TPieceModel);
    destructor Destroy; override;
    procedure RemoveAllFrames;
    procedure ApplyTransformations(AFrame: Single);
    function ApplyTime(Seconds: Single): Integer;
    procedure InterpolateKeyFrames;
    procedure CreatePieceKeyFrames;
    function AddFrame: TFrame;
    procedure SetKeyFrame(AFrame: Integer);
    procedure DeleteKeyFrame(AFrame: Integer);
    procedure InsertFrame(AIndex: Integer);
    procedure DeleteFrame(AIndex: Integer);
    procedure ResetFramePose(AIndex: Integer);
    procedure DeletePieceReferences(APiece: TPiece);
    property Name: string read FName write FName;
    property Model: TPieceModel read FModel;
    property Frame[AIndex: Integer]: TFrame read GetFrame;
    property Frames: Integer read GetFrames write SetFrames;
    property Looping: Boolean read FLooping write SetLooping;
    property Smooth: Boolean read FSmooth write SetSmooth;
    property FPS: Integer read FFPS write FFPS;
  end;

  { TPieceModel }

  TPieceModel = class(TComponent)
  private
    FRoot: TPiece;
    FAnimations: array of TAnimation;
    FBaseRootPosition: TVector;
    function GetAnimations(AIndex: Integer): TAnimation;
    function GetAnimationCount: Integer;
  private
    procedure PieceDestroyed(APiece: TPiece);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SaveToFile(FileName: string);
    procedure LoadFromFile(FileName: string);
    function RayPick(const Ray: TRay): TPiece;
    function AddAnimation(AName: string): TAnimation;
    procedure RemoveAnimation(Animation: TAnimation);
    function IndexOfAnimation(Animation: TAnimation): Integer;
    function FindAnimation(AName: string): TAnimation;
    procedure SaveRootPosition;
    procedure RestoreRootPosition;
    property Root: TPiece read FRoot;
    property BaseRootPosition: TVector read FBaseRootPosition;
    property Animations[AIndex: Integer]: TAnimation read GetAnimations;
    property AnimationCount: Integer read GetAnimationCount;
  end;

implementation

uses
  MeshLoader, LazFileUtils;

{ TFrame }

procedure TFrame.CopyPoseFrom(Src: TFrame);
var
  I: Integer;
begin
  SetLength(KeyFrames, Length(Src.KeyFrames));
  for I:=0 to High(KeyFrames) do
    KeyFrames[I]:=Src.KeyFrames[I];
  RootPos:=Src.RootPos;
end;

{ TAnimation }

function TAnimation.GetFrame(AIndex: Integer): TFrame;
begin
  Result:=FFrames[AIndex];
end;

function TAnimation.GetFrames: Integer;
begin
  Result:=Length(FFrames);
end;

function TAnimation.AddNewFrame: TFrame;
begin
  Result:=TFrame.Create;
  SetLength(FFrames, Length(FFrames) + 1);
  FFrames[High(FFrames)]:=Result;
end;

procedure TAnimation.SetFrames(AValue: Integer);
var
  I: Integer;
begin
  if AValue=Length(FFrames) then Exit;
  if AValue < Length(FFrames) then begin
    for I:=AValue to High(FFrames) do
      FFrames[I].Free;
    SetLength(FFrames, AValue);
  end else begin
    while AValue > Length(FFrames) do AddNewFrame;
    CreatePieceKeyFrames;
  end;
  InterpolateKeyFrames;
end;

procedure TAnimation.SetLooping(AValue: Boolean);
begin
  if FLooping=AValue then Exit;
  FLooping:=AValue;
  InterpolateKeyFrames;
end;

procedure TAnimation.SetSmooth(AValue: Boolean);
begin
  if FSmooth=AValue then Exit;
  FSmooth:=AValue;
  InterpolateKeyFrames;
end;

constructor TAnimation.Create(AModel: TPieceModel);
begin
  FModel:=AModel;
  FFPS:=16;
  FSmooth:=True;
  CreatePieceKeyFrames;
end;

destructor TAnimation.Destroy;
begin
  RemoveAllFrames;
  inherited Destroy;
end;

procedure TAnimation.RemoveAllFrames;
var
  I: Integer;
begin
  for I:=0 to High(FFrames) do FFrames[I].Free;
  SetLength(FFrames, 0);
end;

procedure TAnimation.ApplyTransformations(AFrame: Single);
var
  BaseFrame, I, NextFrame: Integer;
  SubFrame: Single;
  T: TTransform;
begin
  if Frames=0 then Exit;
  BaseFrame:=Trunc(AFrame);
  SubFrame:=AFrame - BaseFrame;
  if BaseFrame < 0 then BaseFrame:=0;
  if Baseframe >= Frames then begin
    if Looping then
      BaseFrame:=BaseFrame mod Frames
    else
      BaseFrame:=Frames - 1;
  end;
  NextFrame:=BaseFrame + 1;
  if NextFrame=Frames then begin
    if Looping then
      NextFrame:=0
    else
      NextFrame:=BaseFrame;
  end;
  T:=Model.Root.Transform;
  T.Translation.x:=FFrames[BaseFrame].RootPos.x*(1-SubFrame) + FFrames[NextFrame].RootPos.x*SubFrame;
  T.Translation.y:=FFrames[BaseFrame].RootPos.y*(1-SubFrame) + FFrames[NextFrame].RootPos.y*SubFrame;
  T.Translation.z:=FFrames[BaseFrame].RootPos.z*(1-SubFrame) + FFrames[NextFrame].RootPos.z*SubFrame;
  Model.Root.Transform:=T;
  for I:=0 to High(FFrames[BaseFrame].KeyFrames) do begin
    T:=FFrames[BaseFrame].KeyFrames[I].Piece.Transform;
    T.Rotation.x:=FFrames[BaseFrame].KeyFrames[I].Rotation.x*(1-SubFrame) + FFrames[NextFrame].KeyFrames[I].Rotation.x*SubFrame;
    T.Rotation.y:=FFrames[BaseFrame].KeyFrames[I].Rotation.y*(1-SubFrame) + FFrames[NextFrame].KeyFrames[I].Rotation.y*SubFrame;
    T.Rotation.z:=FFrames[BaseFrame].KeyFrames[I].Rotation.z*(1-SubFrame) + FFrames[NextFrame].KeyFrames[I].Rotation.z*SubFrame;
    FFrames[BaseFrame].KeyFrames[I].Piece.Transform:=T;
  end;
end;

function TAnimation.ApplyTime(Seconds: Single): Integer;
begin
  if Frames < 1 then Exit;
  if Frames=1 then begin
    ApplyTransformations(0);
    Exit;
  end;
  ApplyTransformations(Seconds*FPS);
  if Looping then
    Result:=Trunc(Seconds*FPS) mod Frames
  else begin
    Result:=Trunc(Seconds*FPS);
    if Result >= Frames then Result:=Frames - 1;
  end;
end;

procedure TAnimation.InterpolateKeyFrames;
var
  I, J, Prev, Next, PrePrev, PostNext: Integer;
  D: Double;
  KeyFrames: array of TKeyFrame;

  function PrevKey(Prev: Integer): Integer; forward;

  function NextKey(Next: Integer): Integer;
  var
    J: Integer;
  begin
    for J:=Next to Frames - 1 do
      if Frame[J].IsKey then Exit(J);
    if Looping then
      Result:=NextKey(0)
    else
      Result:=PrevKey(Frames - 1);
  end;

  function PrevKey(Prev: Integer): Integer;
  var
    J: Integer;
  begin
    for J:=Prev downto 0 do
      if Frame[J].IsKey then Exit(J);
    if Looping then
      Result:=PrevKey(Frames - 1)
    else
      Result:=NextKey(0);
  end;

begin
  CreatePieceKeyFrames;
  J:=0;
  for I:=0 to Frames - 1 do if Frame[I].IsKey then begin
    J:=1;
    Break;
  end;
  if J=0 then Exit;
  for I:=0 to Frames - 1 do if not Frame[I].IsKey then begin
    Prev:=PrevKey(I);
    Next:=NextKey(I);
    if Prev=Next then
      D:=0
    else if Prev < Next then
      D:=(I - Prev)/(Next - Prev)
    else
      D:=(I - Prev)/(Frames - Prev + Next);
    if Smooth then begin
      PrePrev:=PrevKey(Prev - 1);
      PostNext:=NextKey(Next + 1);
      for J:=0 to High(Frame[I].KeyFrames) do begin
        Frame[I].KeyFrames[J].Rotation.x:=
          CatmullRomInterpolate(Frame[PrePrev].KeyFrames[J].Rotation.x,
                           Frame[Prev].KeyFrames[J].Rotation.x,
                           Frame[Next].KeyFrames[J].Rotation.x,
                           Frame[PostNext].KeyFrames[J].Rotation.x, D);
        Frame[I].KeyFrames[J].Rotation.y:=
          CatmullRomInterpolate(Frame[PrePrev].KeyFrames[J].Rotation.y,
                           Frame[Prev].KeyFrames[J].Rotation.y,
                           Frame[Next].KeyFrames[J].Rotation.y,
                           Frame[PostNext].KeyFrames[J].Rotation.y, D);
        Frame[I].KeyFrames[J].Rotation.z:=
          CatmullRomInterpolate(Frame[PrePrev].KeyFrames[J].Rotation.z,
                           Frame[Prev].KeyFrames[J].Rotation.z,
                           Frame[Next].KeyFrames[J].Rotation.z,
                           Frame[PostNext].KeyFrames[J].Rotation.z, D);
      end;
      Frame[I].RootPos.x:=Frame[Prev].RootPos.x*(1-D) + Frame[Next].RootPos.x*D;
      Frame[I].RootPos.y:=Frame[Prev].RootPos.y*(1-D) + Frame[Next].RootPos.y*D;
      Frame[I].RootPos.z:=Frame[Prev].RootPos.z*(1-D) + Frame[Next].RootPos.z*D;
    end else begin
      for J:=0 to High(Frame[I].KeyFrames) do begin
        Frame[I].KeyFrames[J].Rotation.x:=Frame[Prev].KeyFrames[J].Rotation.x*(1-D) + Frame[Next].KeyFrames[J].Rotation.x*D;
        Frame[I].KeyFrames[J].Rotation.y:=Frame[Prev].KeyFrames[J].Rotation.y*(1-D) + Frame[Next].KeyFrames[J].Rotation.y*D;
        Frame[I].KeyFrames[J].Rotation.z:=Frame[Prev].KeyFrames[J].Rotation.z*(1-D) + Frame[Next].KeyFrames[J].Rotation.z*D;
      end;
      Frame[I].RootPos.x:=Frame[Prev].RootPos.x*(1-D) + Frame[Next].RootPos.x*D;
      Frame[I].RootPos.y:=Frame[Prev].RootPos.y*(1-D) + Frame[Next].RootPos.y*D;
      Frame[I].RootPos.z:=Frame[Prev].RootPos.z*(1-D) + Frame[Next].RootPos.z*D;
    end;
  end;
end;

procedure TAnimation.CreatePieceKeyFrames;
var
  FrameIdx: Integer;
  NewKeyFrames: array of TKeyFrame;

  procedure Scan(Piece: TPiece);
  var
    Found, I: Integer;
  begin
    with Frame[FrameIdx] do begin
      Found:=-1;
      for I:=0 to High(KeyFrames) do begin
        if KeyFrames[I].Piece=Piece then begin
          Found:=I;
          Break;
        end;
      end;
      SetLength(NewKeyFrames, Length(NewKeyFrames) + 1);
      NewKeyFrames[High(NewKeyFrames)].Piece:=Piece;
      if Found <> -1 then
        NewKeyFrames[High(NewKeyFrames)].Rotation:=KeyFrames[Found].Rotation
      else
        NewKeyFrames[High(NewKeyFrames)].Rotation:=Piece.Transform.Rotation;
    end;
    for I:=0 to Piece.ChildCount - 1 do Scan(TPiece(Piece.Children[I]));
  end;

begin
  if Frames < 1 then begin
    Frames:=1;
    Exit;
  end;
  FrameIdx:=0;
  while FrameIdx < Frames do begin
    NewKeyFrames:=nil;
    Scan(Model.Root);
    Frame[FrameIdx].KeyFrames:=NewKeyFrames;
    Inc(FrameIdx);
  end;
end;

function TAnimation.AddFrame: TFrame;
begin
  Result:=AddNewFrame;
  CreatePieceKeyFrames;
end;

procedure TAnimation.SetKeyFrame(AFrame: Integer);
var
  I: Integer;
begin
  if AFrame < 0 then Exit;
  if AFrame > Frames then Frames:=AFrame;
  for I:=0 to High(Frame[AFrame].KeyFrames) do
    Frame[AFrame].KeyFrames[I].Rotation:=Frame[AFrame].KeyFrames[I].Piece.Transform.Rotation;
  Frame[AFrame].RootPos:=Model.Root.Transform.Translation;
  Frame[AFrame].IsKey:=True;
  InterpolateKeyFrames;
end;

procedure TAnimation.DeleteKeyFrame(AFrame: Integer);
begin
  if (AFrame < 0) or (AFrame > Frames) then Exit;
  Frame[AFrame].IsKey:=False;
  InterpolateKeyFrames;
end;

procedure TAnimation.InsertFrame(AIndex: Integer);
var
  I: Integer;
begin
  if (AIndex < 0) or (AIndex >= Frames) then Exit;
  SetLength(FFrames, Length(FFrames) + 1);
  for I:=High(FFrames) downto AIndex + 1 do
    FFrames[I]:=FFrames[I - 1];
  FFrames[AIndex]:=TFrame.Create;
  FFrames[AIndex].IsKey:=False;
  SetLength(FFrames[AIndex].KeyFrames, Length(FFrames[AIndex + 1].KeyFrames));
  for I:=0 to High(FFrames[AIndex].KeyFrames) do
    FFrames[AIndex].KeyFrames[I]:=FFrames[AIndex + 1].KeyFrames[I];
  InterpolateKeyFrames;
end;

procedure TAnimation.DeleteFrame(AIndex: Integer);
var
  I: Integer;
begin
  if (AIndex < 0) or (AIndex >= Frames) then Exit;
  FFrames[AIndex].Free;
  for I:=AIndex to High(FFrames) - 1 do
    FFrames[I]:=FFrames[I + 1];
  SetLength(FFrames, Length(FFrames) - 1);
  InterpolateKeyFrames;
end;

procedure TAnimation.ResetFramePose(AIndex: Integer);
var
  I: Integer;
begin
  if (AIndex < 0) or (AIndex >= Frames) then Exit;
  with Frame[AIndex] do begin
    for I:=0 to High(KeyFrames) do
      KeyFrames[I].Rotation.Zero;
    RootPos:=Model.BaseRootPosition;
  end;
end;

procedure TAnimation.DeletePieceReferences(APiece: TPiece);
var
  I, J: Integer;
begin
  for I:=0 to Frames - 1 do with Frame[I] do
    for J:=High(KeyFrames) downto 0 do
      if KeyFrames[J].Piece=APiece then begin
        if J < High(KeyFrames) then
          Move(KeyFrames[J + 1], KeyFrames[J], SizeOf(TKeyFrame)*(Length(KeyFrames) - J - 1));
        SetLength(KeyFrames, Length(KeyFrames) - 1);
      end;
end;

{ TPiece }

constructor TPiece.Create;
begin
  inherited Create;
  FTransform.Reset;
  FWorldMatrix:=IdentityMatrix;
end;

destructor TPiece.Destroy;
begin
  if Assigned(FMesh) then FMesh.Free;
  if Assigned(FModel) then FModel.PieceDestroyed(Self);
  inherited Destroy;
end;

procedure TPiece.SetTransform(AValue: TTransform);
begin
  if FTransform=AValue then Exit;
  FTransform:=AValue;
  UpdateWorldMatrix;
end;

procedure TPiece.ParentChanged;
begin
  inherited ParentChanged;
  UpdateWorldMatrix;
end;

procedure TPiece.ChildAdded(AChild: TNode);
begin
  TPiece(AChild).FModel:=FModel;
end;

procedure TPiece.LoadMesh(AFileName: string);
begin
  FreeAndNil(FMesh);
  FMesh:=LoadMeshFromFile(AFileName, False);
  FMesh.Scale(PieceMeshScale);
  Name:=ExtractFileNameOnly(AFileName);
end;

function TPiece.AddChildPiece(AFileName: string): TPiece;
begin
  Result:=TPiece.Create;
  if AFileName <> '' then Result.LoadMesh(AFileName);
  Add(Result);
end;

procedure TPiece.UpdateWorldMatrix;
var
  I: Integer;
begin
  FWorldMatrix:=Transform.ToMatrix;
  if Parent is TPiece then FWorldMatrix.SwapMultiply(TPiece(Parent).WorldMatrix);
  for I:=0 to ChildCount - 1 do TPiece(Children[I]).UpdateWorldMatrix;
end;

{ TPieceModel }

function TPieceModel.GetAnimationCount: Integer;
begin
  Result:=Length(FAnimations);
end;

procedure TPieceModel.PieceDestroyed(APiece: TPiece);
var
  I: Integer;
begin
  for I:=0 to AnimationCount - 1 do Animations[I].DeletePieceReferences(APiece);
end;

function TPieceModel.GetAnimations(AIndex: Integer): TAnimation;
begin
  Result:=FAnimations[AIndex];
end;

constructor TPieceModel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FRoot:=TPiece.Create;
  FRoot.FModel:=Self;
  SetLength(FAnimations, 1);
  FAnimations[0]:=TAnimation.Create(Self);
  FAnimations[0].Name:='idle';
end;

destructor TPieceModel.Destroy;
var
  I: Integer;
begin
  FreeAndNil(FRoot);
  for I:=0 to High(FAnimations) do FAnimations[I].Free;
  inherited Destroy;
end;

procedure TPieceModel.SaveToFile(FileName: string);
var
  F: File;
  Magic: array [0..3] of Char;

  procedure WritePiece(Piece: TPiece);
  var
    Name: string[31];
    I: Integer;
    S: Single;
  begin
    Name:=Piece.Name;
    BlockWrite(F, Name, 32);
    S:=Piece.Transform.Translation.x/PieceMeshScale; BlockWrite(F, S, 4);
    S:=Piece.Transform.Translation.y/PieceMeshScale; BlockWrite(F, S, 4);
    S:=Piece.Transform.Translation.z/PieceMeshScale; BlockWrite(F, S, 4);
    if Assigned(Piece.Mesh) then I:=Piece.Mesh.VertexCount else I:=0;
    BlockWrite(F, I, 4);
    for I:=0 to I - 1 do begin
      S:=Piece.Mesh.Vertices[I].x/PieceMeshScale; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].y/PieceMeshScale; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].z/PieceMeshScale; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].n.x; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].n.y; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].n.z; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].s; BlockWrite(F, S, 4);
      S:=Piece.Mesh.Vertices[I].t; BlockWrite(F, S, 4);
    end;
    I:=Piece.ChildCount;
    BlockWrite(F, I, 4);
    for I:=0 to I - 1 do WritePiece(TPiece(Piece.Children[I]));
  end;

  procedure WriteAnimations;
  var
    TheName: string[31];
    I, J, K: Integer;
    B: Byte;
    W: Word;
    S: Single;
  begin
    I:=AnimationCount;
    BlockWrite(F, I, 4);
    for I:=0 to AnimationCount - 1 do with Animations[I] do begin
      TheName:=Name;
      BlockWrite(F, TheName, 32);
      B:=0;
      if Looping then B:=1;
      BlockWrite(F, B, 1);
      B:=FPS;
      BlockWrite(F, FPS, 1);
      W:=Frames;
      BlockWrite(F, W, 2);
      for J:=0 to Frames - 1 do with Frame[J] do begin
        B:=0;
        if IsKey then B:=1;
        if IsEvent then B:=B or 2;
        BlockWrite(F, B, 1);
        S:=RootPos.x/PieceMeshScale; BlockWrite(F, S, 4);
        S:=RootPos.y/PieceMeshScale; BlockWrite(F, S, 4);
        S:=RootPos.z/PieceMeshScale; BlockWrite(F, S, 4);
        for K:=0 to High(KeyFrames) do begin
          S:=KeyFrames[K].Rotation.x; BlockWrite(F, S, 4);
          S:=KeyFrames[K].Rotation.y; BlockWrite(F, S, 4);
          S:=KeyFrames[K].Rotation.z; BlockWrite(F, S, 4);
        end;
      end;
    end;
  end;

begin
  AssignFile(F, FileName);
  {$I-}
  ReWrite(F, 1);
  {$I+}
  if IOResult <> 0 then Exit;
  Magic[0]:='P';
  Magic[1]:='M';
  Magic[2]:='F';
  Magic[3]:='1';
  BlockWrite(F, Magic, 4);
  WritePiece(Root);
  WriteAnimations;
  Close(F);
end;

procedure TPieceModel.LoadFromFile(FileName: string);
var
  F: File;
  Magic: array [0..3] of Char;
  LoadedPieces: array of TPiece;
  I: Integer;

  procedure ReadPiece(Piece: TPiece);
  var
    Name: string[31];
    S: Single;
    T: TTransform;
    V, N: TVector;
    TS, TT: Double;
    A, B, C: Integer;
    I: Integer;
    Child: TPiece;
  begin
    S:=0;
    BlockRead(F, Name, 32);
    Piece.Name:=Name;
    T.Reset;
    BlockRead(F, S, 4); T.Translation.x:=S*PieceMeshScale;
    BlockRead(F, S, 4); T.Translation.y:=S*PieceMeshScale;
    BlockRead(F, S, 4); T.Translation.z:=S*PieceMeshScale;
    Piece.Transform:=T;
    BlockRead(F, I, 4);
    if I > 0 then begin
      Piece.FMesh:=TMesh.Create;
      for I:=0 to (I div 3)-1 do begin
        BlockRead(F, S, 4); V.x:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.y:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.z:=S*PieceMeshScale;
        BlockRead(F, S, 4); N.x:=S;
        BlockRead(F, S, 4); N.y:=S;
        BlockRead(F, S, 4); N.z:=S;
        BlockRead(F, S, 4); TS:=S;
        BlockRead(F, S, 4); TT:=S;
        A:=Piece.Mesh.AddVertex(V, N, TS, TT);
        BlockRead(F, S, 4); V.x:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.y:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.z:=S*PieceMeshScale;
        BlockRead(F, S, 4); N.x:=S;
        BlockRead(F, S, 4); N.y:=S;
        BlockRead(F, S, 4); N.z:=S;
        BlockRead(F, S, 4); TS:=S;
        BlockRead(F, S, 4); TT:=S;
        B:=Piece.Mesh.AddVertex(V, N, TS, TT);
        BlockRead(F, S, 4); V.x:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.y:=S*PieceMeshScale;
        BlockRead(F, S, 4); V.z:=S*PieceMeshScale;
        BlockRead(F, S, 4); N.x:=S;
        BlockRead(F, S, 4); N.y:=S;
        BlockRead(F, S, 4); N.z:=S;
        BlockRead(F, S, 4); TS:=S;
        BlockRead(F, S, 4); TT:=S;
        C:=Piece.Mesh.AddVertex(V, N, TS, TT);
        Piece.Mesh.AddFace(A, B, C);
      end;
    end;
    BlockRead(F, I, 4);
    for I:=0 to I - 1 do begin
      Child:=TPiece.Create;
      SetLength(LoadedPieces, Length(LoadedPieces) + 1);
      LoadedPieces[High(LoadedPieces)]:=Child;
      Piece.Add(Child);
      ReadPiece(Child);
    end;
  end;

  procedure ReadAnimations;
  var
    Anim: TAnimation;
    TheName: string[31];
    I, J, K: Integer;
    B: Byte;
    W: Word;
    S: Single;
  begin
    BlockRead(F, I, 4);
    for I:=0 to I - 1 do begin
      Anim:=TAnimation.Create(Self);
      SetLength(FAnimations, Length(FAnimations) + 1);
      FAnimations[High(FAnimations)]:=Anim;
      with Anim do begin
        RemoveAllFrames;
        BlockRead(F, TheName, 32);
        FName:=TheName;
        BlockRead(F, B, 1);
        FLooping:=(B and 1)=1;
        BlockRead(F, B, 1);
        FFPS:=B;
        BlockRead(F, W, 2);
        SetLength(FFrames, W);
        for J:=0 to W - 1 do FFrames[J]:=TFrame.Create;
        for J:=0 to W - 1 do with FFrames[J] do begin
          SetLength(KeyFrames, Length(LoadedPieces));
          BlockRead(F, B, 1);
          IsKey:=(B and 1)=1;
          IsEvent:=(B and 2)=2;
          BlockRead(F, S, 4); RootPos.x:=S*PieceMeshScale;
          BlockRead(F, S, 4); RootPos.y:=S*PieceMeshScale;
          BlockRead(F, S, 4); RootPos.z:=S*PieceMeshScale;
          for K:=0 to High(KeyFrames) do begin
            KeyFrames[K].Piece:=LoadedPieces[K];
            BlockRead(F, S, 4); KeyFrames[K].Rotation.x:=S;
            BlockRead(F, S, 4); KeyFrames[K].Rotation.y:=S;
            BlockRead(F, S, 4); KeyFrames[K].Rotation.z:=S;
          end;
        end;
        InterpolateKeyFrames;
      end;
    end;
  end;

begin
  FRoot.Free;
  for I:=0 to High(FAnimations) do FAnimations[I].Free;
  FAnimations:=nil;
  FRoot:=TPiece.Create;
  FRoot.FModel:=Self;
  AssignFile(F, FileName);
  {$I-}
  Reset(F, 1);
  {$I+}
  if IOResult <> 0 then Exit;
  Magic:='    ';
  BlockRead(F, Magic, 4);
  if Magic <> 'PMF1' then begin
    CloseFile(F);
    Exit;
  end;
  SetLength(LoadedPieces, 1);
  LoadedPieces[0]:=Root;
  ReadPiece(Root);
  Root.UpdateWorldMatrix;
  SaveRootPosition;
  ReadAnimations;
  CloseFile(F);
end;

function TPieceModel.RayPick(const Ray: TRay): TPiece;
var
  D, BD: Single;
  IP: TVector;
  R: TPiece;

  procedure Scan(Piece: TPiece);
  var
    I: Integer;
  begin
    if Assigned(Piece.Mesh) then begin
      if Piece.Mesh.RayHit(Ray, Piece.WorldMatrix, IP) then begin
        D:=DistanceSq(IP, Ray.o);
        if D < BD then begin
          R:=Piece;
          BD:=D;
        end;
      end;
    end;
    for I:=0 to Piece.ChildCount - 1 do
      Scan(TPiece(Piece.Children[I]));
  end;

begin
  R:=nil;
  BD:=$FFFFFFFF;
  Scan(Root);
  Result:=R;
end;

function TPieceModel.AddAnimation(AName: string): TAnimation;
begin
  AName:=Trim(AName);
  if Assigned(FindAnimation(AName)) then Exit;
  Result:=TAnimation.Create(Self);
  Result.Name:=AName;
  SetLength(FAnimations, Length(FAnimations) + 1);
  FAnimations[High(FAnimations)]:=Result;
end;

procedure TPieceModel.RemoveAnimation(Animation: TAnimation);
var
  Index, I: Integer;
begin
  Index:=IndexOfAnimation(Animation);
  if Index=-1 then Exit;
  FAnimations[Index].Free;
  for I:=Index to High(FAnimations) - 1 do
    FAnimations[I]:=FAnimations[I + 1];
  SetLength(FAnimations, Length(FAnimations) - 1);
end;

function TPieceModel.IndexOfAnimation(Animation: TAnimation): Integer;
var
  I: Integer;
begin
  for I:=0 to High(FAnimations) do
    if FAnimations[I]=Animation then Exit(I);
  Result:=-1;
end;

function TPieceModel.FindAnimation(AName: string): TAnimation;
var
  I: Integer;
begin
  for I:=0 to High(FAnimations) do
    if LowerCase(AName)=LowerCase(FAnimations[I].Name) then Exit(FAnimations[I]);
  Result:=nil;
end;

procedure TPieceModel.SaveRootPosition;
begin
  FBaseRootPosition:=Root.Transform.Translation;
end;

procedure TPieceModel.RestoreRootPosition;
var
  T: TTransform;
begin
  T:=Root.Transform;
  T.Translation:=BaseRootPosition;
  Root.Transform:=T;
end;

end.

