#include "win32winsys.h"

static HCURSOR cursors[FF_CROSSHAIR + 1];

void* ff_ws_create_window(void* parent, int x, int y, int width, int height, unsigned flags)
{
    wswin_t* win = calloc(1, sizeof(wswin_t));
    wswin_t* pwin = parent;
    LPCTSTR class_name = TEXT("FFWSWIN");
    DWORD style = WS_CLIPCHILDREN|WS_CLIPSIBLINGS;
    RECT r;

    win->parent = pwin;

    if (width < 1) width = 1;
    if (height < 1) height = 1;

    if (flags & FF_OPENGL)
        class_name = TEXT("FFWSWINGL");

    if (!parent || (flags & FF_POPUP)) {
        style |= WS_SYSMENU;
        if ((flags & FF_NOFRAME) == 0) {
            style |= WS_MINIMIZEBOX|WS_CAPTION;
            if ((flags & FF_NORESIZE) == 0)
                style |= WS_SIZEBOX|WS_MAXIMIZEBOX;
        }
        if (flags & FF_POPUP)
            style |= WS_POPUP;
        else {
            x = CW_USEDEFAULT;
            y = CW_USEDEFAULT;
        }
    } else {
        style |= WS_CHILD;
    }

    r.left = r.top = 0;
    r.right = width;
    r.bottom = height;
    AdjustWindowRect(&r, style, FALSE);
    win->w = CreateWindowEx(
        ((WS_CAPTION & style) && (FF_POPUP & flags)) ? WS_EX_TOOLWINDOW : 0,
        class_name, TEXT(""),
        style,
        x, y, r.right - r.left, r.bottom - r.top,
        pwin ? pwin->w : NULL, NULL,
        GetModuleHandle(NULL),
        NULL);
    if (!win->w) goto fail;
    win->style = style;

    if (flags & FF_OPENGL) {
        PIXELFORMATDESCRIPTOR pfd;
        int pf;
        win->dbdc = GetDC(win->w);
        if (!win->dbdc) goto fail;
        pfd.nSize = sizeof(pfd);
        pfd.nVersion = 1;
        pfd.dwFlags = PFD_DRAW_TO_WINDOW|PFD_SUPPORT_OPENGL|PFD_DOUBLEBUFFER;
        pfd.iPixelType = PFD_TYPE_RGBA;
        pfd.cColorBits = 24;
        pfd.cAlphaBits = 8;
        pfd.cDepthBits = 24;
        pfd.cStencilBits = 8;
        pf = ChoosePixelFormat(win->dbdc, &pfd);
        if (!pf) goto fail;
        if (!SetPixelFormat(win->dbdc, pf, &pfd)) goto fail;
        win->rc = wglCreateContext(win->dbdc);
        if (!win->rc) goto fail;
    }

    if (!parent && (flags & FF_CENTERED) && !(flags & FF_INVISIBLE)) {
        RECT dr;
        GetClientRect(GetDesktopWindow(), &dr);
        ff_ws_move(win, (dr.right - (r.right - r.left)) / 2,
                        (dr.bottom - (r.bottom - r.top)) / 2);
    }

    ff_ws_regwin((void*)win->w, win);

    if (!(flags & FF_INVISIBLE))
        ShowWindow(win->w, (FF_POPUP & flags) ? SW_SHOWNA : SW_SHOW);

    win->flags = flags;

    return win;

fail:
    if (win->dbdc) ReleaseDC(win->w, win->dbdc);
    if (win->w) DestroyWindow(win->w);
    free(win);
    return NULL;
}

void ff_ws_destroy_window(void* window)
{
    wswin_t* win = window;
    if (!win) return;
    if (win->flags & FF_OPENGL) {
        if (win->rc) wglDeleteContext(win->rc);
        ReleaseDC(win->w, win->dbdc);
    }
    if (win->gc) ff_ws_free_gc(win->gc, win);
    if (win->db) DeleteObject(win->db);
    DestroyWindow(win->w);
}

void ff_ws_set_caption(void* window, const char* caption)
{
    wswin_t* win = window;
#ifdef UNICODE
    unsigned short* utf16;
#endif
    if (!win) return;
#ifdef UNICODE
    utf16 = ff_8to16((const unsigned char*)caption);
    SetWindowText(win->w, utf16);
    free(utf16);
#else
    if (!ff_isascii(caption)) {
        char* ascii = ff_asciify(caption);
        SetWindowText(win->w, ascii);
        free(ascii);
    } else
        SetWindowText(win->w, caption);
#endif
}

void ff_ws_show(void* window, int show)
{
    wswin_t* win = window;
    if (!win) return;
    if (show)
        ShowWindow(win->w, win->flags & FF_POPUP ? SW_SHOWNA : SW_SHOW);
    else
        ShowWindow(win->w, SW_HIDE);
}

int ff_ws_shown(void* window)
{
    wswin_t* win = window;
    if (!win) return FF_NO;
    return (GetWindowLong(win->w, GWL_STYLE)&WS_VISIBLE) ? FF_YES : FF_NO;
}

void ff_ws_set_ffwindow(void* window, ff_window_t ffwindow)
{
    wswin_t* win = window;
    if (!win) return;
    win->ffwindow = ffwindow;
}

ff_window_t ff_ws_get_ffwindow(void* window)
{
    wswin_t* win = window;
    if (!win) return NULL;
    return win->ffwindow;
}

void ff_ws_move(void* window, int x, int y)
{
    wswin_t* win = window;
    if (!win) return;
    SetWindowPos(win->w, NULL, x, y, 0, 0, SWP_NOCOPYBITS|SWP_NOACTIVATE|SWP_NOOWNERZORDER|SWP_NOSIZE|SWP_NOZORDER);
}

void ff_ws_resize(void* window, int width, int height)
{
    wswin_t* win = window;
    RECT r;
    if (!win) return;
    if (width < 1) width = 1;
    if (height < 1) height = 1;
    r.left = r.top = 0;
    r.right = width;
    r.bottom = height;
    AdjustWindowRect(&r, win->style, FALSE);
    SetWindowPos(win->w, NULL, 0, 0, r.right - r.left, r.bottom - r.top, SWP_NOCOPYBITS|SWP_NOACTIVATE|SWP_NOOWNERZORDER|SWP_NOMOVE|SWP_NOZORDER);
}

void ff_ws_paint(void* window)
{
    wswin_t* win = window;
    if (!win) return;
    InvalidateRect(win->w, NULL, FALSE);
}

void ff_ws_damage(void* window, int x1, int y1, int x2, int y2)
{
    wswin_t* win = window;
    RECT r;
    if (!win || x1 > x2 || y1 > y2) return;
    r.left = x1;
    r.top = y1;
    r.right = x2 + 1;
    r.bottom = y2 + 1;
    InvalidateRect(win->w, &r, FALSE);
}

void ff_damaged(ff_gc_t gc, int* x1, int* y1, int* x2, int* y2)
{
    if (gc->w) {
        RECT r;
        GetUpdateRect(gc->w, &r, FALSE);
        if (x1) *x1 = r.left;
        if (y1) *y1 = r.top;
        if (x2) *x2 = r.right - 1;
        if (y2) *y2 = r.bottom - 1;
    } else {
        if (x1) *x1 = 0;
        if (y1) *y1 = 0;
        if (x2) *x2 = gc->bmpw;
        if (y2) *y2 = gc->bmph;
    }
}

int ff_isdamaged(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (x1 <= x2 && y1 <= y2) {
        if (gc->w) {
            RECT r;
            r.left = x1;
            r.top = y1;
            r.right = x2 + 1;
            r.bottom = y2 + 1;
            return RectVisible(gc->dc, &r) ? FF_YES : FF_NO;
        } else return FF_YES;
    } else return FF_NO;
}

void ff_ws_translate(int* x, int* y, void* from, void* to)
{
    HWND src = from ? ((wswin_t*)from)->w : GetDesktopWindow();
    HWND dst = to ? ((wswin_t*)to)->w : GetDesktopWindow();
    POINT p;
    p.x = x ? *x : 0;
    p.y = y ? *y : 0;
    MapWindowPoints(src, dst, &p, 1);
    if (x) *x = p.x;
    if (y) *y = p.y;
}

void ff_ws_area(void* window, int* x, int* y, int* width, int* height)
{
    wswin_t* win = window;
    RECT r;
    if (!win) return;
    if (x || y) {
        GetWindowRect(win->w, &r);
        if (x) *x = r.left;
        if (y) *y = r.top;
    }
    if (width || height) {
        GetClientRect(win->w, &r);
        if (width) *width = r.right;
        if (height) *height = r.bottom;
    }
}

void* ff_ws_pick(void* window, int x, int y)
{
    HWND win, pwin;
    POINT p;
    p.x = x;
    p.y = y;
    if (window)
        ClientToScreen(((wswin_t*)window)->w, &p);
    win = WindowFromPoint(p);
    if (!window)
        return ff_ws_wswin((void*)win);
    pwin = ((wswin_t*)window)->w;
    while (pwin) {
        if (pwin == win)
            return ff_ws_wswin((void*)win);
        pwin = GetParent(pwin);
    }
    return NULL;
}

void ff_ws_pointer(int* x, int* y)
{
    POINT p;
    GetCursorPos(&p);
    if (x) *x = p.x;
    if (y) *y = p.y;
}

void ff_ws_focus(void* window)
{
    HWND win = window ? ((wswin_t*)window)->w : NULL;
    if (!win) return;
    SetFocus(win);
}

int ff_ws_focused(void* window)
{
    HWND win = window ? ((wswin_t*)window)->w : NULL;
    if (!win) return 0;
    return win == GetFocus();
}

void ff_ws_set_icon(void* window, ff_bitmap_t icon)
{
}

void ff_ws_cursor(void* window, ff_uintptr_t cursor)
{
    static const LPCSTR map[sizeof(cursors)/sizeof(cursors[0])] = {
        0,                      /* FF_NOTHING */
        IDC_ARROW,              /* FF_ARROW */
        IDC_SIZEWE,             /* FF_HARROWS */
        IDC_SIZENS,             /* FF_VARROWS */
        IDC_SIZENWSE,           /* FF_TLBRARROWS */
        IDC_SIZENESW,           /* FF_TRBLARROWS */
        IDC_SIZEALL,            /* FF_4DARROWS */
        IDC_WAIT,               /* FF_WAIT */
        IDC_NO,                 /* FF_UNAVAILABLE */
        IDC_IBEAM,              /* FF_IBEAM */
        IDC_CROSS
    };
    wswin_t* wswin = window;
    if (!wswin || !wswin->w ||
        cursor >= sizeof(cursors)/sizeof(cursors[0]))
        return;
    if (cursor > 0 && !cursors[cursor])
        cursors[cursor] = LoadCursor(NULL, map[cursor]);
    wswin->cursor = cursors[cursor];
}

static void send_enable_event(ff_window_t win, int enable)
{
    ff_window_t* children;
    int i;
    ff_post(win, FF_ENABLE, enable, 0, 0, NULL, FF_NOFLAGS);
    ff_paint(win);
    children = ff_npchildren(win);
    for (i=0; children[i]; i++)
        send_enable_event(children[i], enable);
    ff_free_children(children);
}

void ff_ws_enable(void* window, int enable)
{
    wswin_t* wswin = window;
    if (!wswin || !wswin->w) return;
    if ((!enable && (wswin->wsflags & FF_WSFLAG_DISABLED)) ||
        (enable && !(wswin->wsflags & FF_WSFLAG_DISABLED))) return;
    if (enable) wswin->wsflags &= ~FF_WSFLAG_DISABLED;
    else wswin->wsflags |= FF_WSFLAG_DISABLED;
    send_enable_event(wswin->ffwindow, enable ? FF_YES : FF_NO);
}

int ff_ws_enabled(void* window)
{
    wswin_t* wswin = window;
    if (!wswin || !wswin->w) return 0;
    if (wswin->parent && !(wswin->flags & FF_POPUP) &&
        !ff_ws_enabled(wswin->parent)) return FF_NO;
    return !(wswin->wsflags & FF_WSFLAG_DISABLED);
}
