#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

#define ALIGN_START 0
#define ALIGN_MIDDLE 1
#define ALIGN_END 2

typedef struct _state_t
{
    char* caption;
    int halign, valign;
} state_t;

static void update_prefsize(ff_window_t lbl)
{
    state_t* state = ff_get(lbl, "-state");
    int tw, th;
    ff_font_text_size(ff_font_of(lbl), state->caption, &tw, &th);
    ff_prefsize(lbl, tw, th + 2);
}

static int set_handler(void* lbl, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(lbl, "-state");
    if (!strcmp(nv->name, "caption")) {
        free(state->caption);
        state->caption = ff_strdup(nv->value);
        update_prefsize(lbl);
        ff_paint(lbl);
        return 1;
    } else if (!strcmp(nv->name, "halign")) {
        const char* value = nv->value;
        if (!value || !strcmp(value, "left"))
            state->halign = ALIGN_START;
        else if (!strcmp(value, "center"))
            state->halign = ALIGN_MIDDLE;
        else if (!strcmp(value, "right"))
            state->halign = ALIGN_END;
        ff_paint(lbl);
        return 1;
    } else if (!strcmp(nv->name, "valign")) {
        const char* value = nv->value;
        if (!value || !strcmp(value, "top"))
            state->valign = ALIGN_START;
        else if (!strcmp(value, "center"))
            state->valign = ALIGN_MIDDLE;
        else if (!strcmp(value, "bottom"))
            state->valign = ALIGN_END;
        ff_paint(lbl);
        return 1;
    } else if (!strcmp(nv->name, "talign")) {
        const char* value = nv->value;
        if (!value || !strcmp(value, "top"))
            state->halign = state->valign = ALIGN_START;
        else if (!strcmp(value, "center"))
            state->halign = state->valign = ALIGN_MIDDLE;
        else if (!strcmp(value, "bottom"))
            state->halign = state->valign = ALIGN_END;
        ff_paint(lbl);
        return 1;
    }
    return 0;
}

static int get_handler(void* lbl, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(lbl, "-state");
    if (!strcmp(nv->name, "caption")) {
        nv->value = state->caption;
        return 1;
    }
    return 0;
}

static int destroy_handler(void* lbl, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(lbl, "-state");
    free(state->caption);
    free(state);
    return 0;
}

static int newfont_handler(void* lbl, ff_event_t* ev, void* data)
{
    update_prefsize(lbl);
    return 0;
}

static int paint_handler(void* lbl, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th, tx, ty;
    state_t* state = ff_get(lbl, "-state");
    ff_area(lbl, NULL, NULL, &w, &h);
    ff_color_attr(gc, lbl, "background", FF_3DFACE_COLOR);
    ff_fill(gc, 0, 0, w, h);
    ff_text_size(gc, state->caption, &tw, &th);

    switch (state->halign) {
    case ALIGN_START:
        tx = 0;
        break;
    case ALIGN_MIDDLE:
        tx = ff_round_int(((float)w - (float)tw) / 2.0f);
        break;
    case ALIGN_END:
        tx = w - tw;
        break;
    default: return 1;
    }
    switch (state->valign) {
    case ALIGN_START:
        ty = th;
        break;
    case ALIGN_MIDDLE:
        ty = ff_round_int(((float)h + (float)th) / 2.0f);
        break;
    case ALIGN_END:
        ty = h - th;
        break;
    default: return 1;
    }
    ff_color_attr(gc, lbl, "foreground",
        ff_enabled(lbl) ? FF_3DTEXT_COLOR : FF_3DDISABLEDTEXT_COLOR);
    ff_text(gc, tx, ty, state->caption);
    return 1;
}

ff_window_t ff_label(ff_window_t parent, const char* caption)
{
    ff_window_t lbl = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    ff_set(lbl, "-state", state);
    ff_set(lbl, "class", (void*)"label");
    ff_link(lbl, FF_SET, set_handler, NULL);
    ff_link(lbl, FF_GET, get_handler, NULL);
    ff_set(lbl, "halign", "left");
    ff_set(lbl, "valign", "center");
    ff_set(lbl, "caption", (void*)caption);
    ff_link(lbl, FF_DESTROY, destroy_handler, NULL);
    ff_link(lbl, FF_NEWFONT, newfont_handler, NULL);
    ff_link(lbl, FF_PAINT, paint_handler, NULL);
    return lbl;
}

