#include <lforms.h>

struct fontinfo {
    char* name;
    unsigned flags;
    int* size;
    int sizes;
};

static ff_window_t main_frame;  /* The main frame (window) */
static ff_window_t example;     /* The example label */
static ff_window_t names;       /* The font names list */
static ff_window_t styles;      /* The font styles list */
static ff_window_t sizes;       /* The font sizes list */
static struct fontinfo* info;   /* Font information array */
static size_t infos;            /* Elements in the info array */
static const char* show_name;   /* The name of the font to show */
static unsigned show_flags;     /* The flags to use for the font */
static int show_size;           /* The size of the font to show */

/* Update the example font, called by the list selection handlers */
static void update_example_font(void)
{
    ff_font_t font = ff_font_search(show_name, show_size, show_flags);
    ff_set(example, "font", font);
}

/* Called for each available font */
static int font_enum_proc(const char* name, const int* size, int sizes, unsigned flags, void* data)
{
    /* Add the information to the font info array */
    info = realloc(info, sizeof(struct fontinfo)*(infos + 1));
    info[infos].name = ff_strdup(name);
    info[infos].size = malloc(sizeof(int)*(size_t)sizes);
    memcpy(info[infos].size, size, sizeof(int)*(size_t)sizes);
    info[infos].sizes = sizes;
    info[infos].flags = flags;
    infos++;
    /* Continue with the enumeration */
    return FF_YES;
}

/* Called when a name is selected in the names list */
static int names_select_handler(void* win, ff_event_t* ev, void* data)
{
    struct fontinfo* fi = info + ev->x;
    int i;

    /* Fill the font styles */
    ff_freeze(styles);
    ff_list_clear(styles);
    ff_list_add(styles, "Normal", NULL);
    ff_list_add(styles, "Underline", (void*)FF_UNDERLINE);
    ff_list_add(styles, "Strikeout", (void*)FF_STRIKEOUT);
    if (fi->flags & FF_ITALIC) {
        ff_list_add(styles, "Italic", (void*)FF_ITALIC);
        ff_list_add(styles, "Italic Underline", (void*)(FF_ITALIC|FF_UNDERLINE));
        ff_list_add(styles, "Italic Strikeout", (void*)(FF_ITALIC|FF_STRIKEOUT));
    }
    if (fi->flags & FF_BOLD) {
        ff_list_add(styles, "Bold", (void*)FF_BOLD);
        ff_list_add(styles, "Bold Underline", (void*)(FF_BOLD|FF_UNDERLINE));
        ff_list_add(styles, "Bold Strikeout", (void*)(FF_BOLD|FF_STRIKEOUT));
        if (fi->flags & FF_ITALIC) {
            ff_list_add(styles, "Italic Bold", (void*)(FF_ITALIC|FF_BOLD));
            ff_list_add(styles, "Italic Bold Underline", (void*)(FF_ITALIC|FF_BOLD|FF_UNDERLINE));
            ff_list_add(styles, "Italic Bold Strikeout", (void*)(FF_ITALIC|FF_BOLD|FF_STRIKEOUT));
        }
    }
    ff_thaw(styles);

    /* Fill the font sizes */
    ff_freeze(sizes);
    ff_list_clear(sizes);
    for (i=0; i < fi->sizes; i++) {
        char tmp[64];
        sprintf(tmp, "%i", fi->size[i]);
        ff_list_add(sizes, tmp, NULL);
    }
    ff_thaw(sizes);

    /* Update the example font */
    show_name = fi->name;
    update_example_font();

    return FF_YES;
}

/* Called when a style is selected */
static int styles_select_handler(void* win, ff_event_t* ev, void* data)
{
    if (ev->x >= 0) {
        show_flags = (unsigned)(ff_uintptr_t)ff_list_data(styles, ev->x);
        update_example_font();
    }
    return FF_YES;
}

/* Called when a size is selected */
static int sizes_select_handler(void* win, ff_event_t* ev, void* data)
{
    if (ev->x >= 0) {
        show_size = atoi(ff_list_text(sizes, ev->x));
        update_example_font();
    }
    return FF_YES;
}

/* Release the memory for the font information array */
static void free_font_info(void)
{
    size_t i;
    for (i=0; i < infos; i++) {
        free(info[i].name);
        free(info[i].size);
    }
    free(info);
}

/* Entry point */
int main(int argc, char** argv)
{
    ff_window_t example_container, column_container, column, header,
                label;
    size_t i;
    int r;

    /* Initialize */
    if (!ff_init(argc, argv, "Little Forms Fonts"))
        return 1;

    /* Create main frame in a frozen state to avoid flickering */
    main_frame = ff_mainframe("Little Forms Fonts Example", NULL,
                              450, 250,
                              FF_FROZEN);
    ff_layout(main_frame, ff_border);
    ff_seti(main_frame, "padding-around", 10);

    /* Create the example */
    example_container = ff_box(main_frame, NULL, "thick-sunken");
    ff_set(example_container, "border-align", "top");
    ff_seti(example_container, "minimum-height", 64);
    ff_seti(example_container, "margin-bottom", 2);
    ff_seti(example_container, "background", FF_3DLIGHT_COLOR);
    ff_layout(example_container, ff_border);
    example = ff_label(example_container, "Sphinx of black quartz, judge my vow.");
    ff_seti(example, "background", FF_3DLIGHT_COLOR);
    ff_seti(example, "foreground", FF_3DTEXT_COLOR);
    ff_set(example, "border-align", "center");
    ff_set(example, "halign", "center");

    /* Create container for the three columns */
    column_container = ff_box(main_frame, NULL, "none");
    ff_set(column_container, "border-align", "center");
    ff_layout(column_container, ff_stack);
    ff_set(column_container, "stack-orientation", "horizontal");
    ff_set(column_container, "stack-align", "fill");
    ff_seti(column_container, "stack-spacing", 4);

    /* Create the font name column */
    column = ff_box(column_container, NULL, "none");
    ff_layout(column, ff_border);
    header = ff_box(column, NULL, "thick-sunken");
    ff_set(header, "border-align", "top");
    ff_layout(header, ff_border);
    ff_seti(header, "background", FF_3DSHADOW_COLOR);
    label = ff_label(header, "Font");
    ff_seti(label, "background", FF_3DSHADOW_COLOR);
    ff_seti(label, "foreground", FF_3DLIGHT_COLOR);
    ff_set(label, "font", ff_default_font(FF_BOLD_FONT));
    ff_set(label, "halign", "center");
    names = ff_list(column, NULL, NULL, 0, 0);
    ff_seti(names, "margin-top", 2);
    ff_set(names, "border-align", "center");
    ff_link(names, FF_SELECT, names_select_handler, NULL);

    /* Create the styles column */
    column = ff_box(column_container, NULL, "none");
    ff_layout(column, ff_border);
    ff_seti(column, "stack-noscale", FF_YES);
    ff_seti(column, "minimum-width", 170);
    header = ff_box(column, NULL, "thick-sunken");
    ff_set(header, "border-align", "top");
    ff_layout(header, ff_border);
    ff_seti(header, "background", FF_3DSHADOW_COLOR);
    label = ff_label(header, "Style");
    ff_seti(label, "background", FF_3DSHADOW_COLOR);
    ff_seti(label, "foreground", FF_3DLIGHT_COLOR);
    ff_set(label, "font", ff_default_font(FF_BOLD_FONT));
    ff_set(label, "halign", "center");
    styles = ff_list(column, NULL, NULL, 0, 0);
    ff_seti(styles, "margin-top", 2);
    ff_set(styles, "border-align", "center");
    ff_link(styles, FF_SELECT, styles_select_handler, NULL);

    /* Create the size column */
    column = ff_box(column_container, NULL, "none");
    ff_layout(column, ff_border);
    ff_seti(column, "stack-noscale", FF_YES);
    ff_seti(column, "minimum-width", 60);
    header = ff_box(column, NULL, "thick-sunken");
    ff_set(header, "border-align", "top");
    ff_layout(header, ff_border);
    ff_seti(header, "background", FF_3DSHADOW_COLOR);
    label = ff_label(header, "Size");
    ff_seti(label, "background", FF_3DSHADOW_COLOR);
    ff_seti(label, "foreground", FF_3DLIGHT_COLOR);
    ff_set(label, "font", ff_default_font(FF_BOLD_FONT));
    ff_set(label, "halign", "center");
    sizes = ff_list(column, NULL, NULL, 0, 0);
    ff_seti(sizes, "margin-top", 2);
    ff_set(sizes, "border-align", "center");
    ff_link(sizes, FF_SELECT, sizes_select_handler, NULL);

    /* Enumerate available fonts */
    ff_font_enum(font_enum_proc, NULL, FF_NOFLAGS);

    /* Fill the fonts list */
    for (i=0; i < infos; i++)
        ff_list_add(names, info[i].name, NULL);

    /* Fill the default values for the show_ variables */
    show_name = ff_font_name(NULL);
    show_flags = ff_font_flags(NULL);
    show_size = ff_font_size(NULL);
    update_example_font();

    /* Select the first font (the linked callback for the FF_SELECT
     * that this call will emit will fill the Style and Size lists) */
    if (infos > 0) ff_list_select(names, 0, FF_YES);

    /* Thaw main frame */
    ff_thaw(main_frame);

    /* Enter main loop */
    r = ff_run();

    /* Shutdown */
    free_font_info();
    return r;
}
